/*
 * THIS SOFTWARE IS PROVIDED BY THE CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL M2J OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */ 


package M2J;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;



public class UDF {
    
   
    /**
     * Return the first match of string that matches the pattern.
     * @param pattern represents the pattern of the string you want to find
     * @param searchString is the string you want to parse
     * @return boolean indicating if specific IBAN has a valid check digit
     */     
    static String findFirstMatch(String pattern, String searchString) {
        // if the pattern contains one or more '#' signs, replace each with '\d' 
        // '\d' is Java's pattern wild card for any digit 
        pattern = pattern.replaceAll("#", "\\\\d"); 
        Pattern p = Pattern.compile(pattern.trim());
        Matcher m = p.matcher(searchString.trim());
        if (m.find()) 
            return m.group(); // return the matching part of the string
        else
            return ""; // return an empty string if no match can be found
    }
    
    
    /**
     * Return the last match of string that matches the pattern.
     * @param pattern represents the pattern of the string you want to find
     * @param searchString is the string you want to parse
     * @return boolean indicating if specific IBAN has a valid check digit
     */    
    static String findLastMatch(String pattern, String searchString) {
        // if the pattern contains one or more '#' signs, replace each with '\d' 
        // '\d' is Java's pattern wild card for any digit 
        pattern = pattern.replaceAll("#", "\\\\d"); 
        Pattern p = Pattern.compile(pattern.trim());
        Matcher m = p.matcher(searchString.trim());
        String result = "";
        while (m.find()) 
            result = m.group(); // set the result to the matched part of the string
        return result; // return an empty string if no match can be found
    }   
    
    
    /**
     * Lists the running Windows processes.
     * requires tasklist.exe, which is bundled with most windows operating systems.
     * @return Comma separated String of process names
     */    
    static String listRunningProcesses() {
        String processes = "";
        try {
          String line;
          Process p = Runtime.getRuntime().exec("tasklist.exe /fo csv /nh");
          BufferedReader input = new BufferedReader(new InputStreamReader(p.getInputStream()));
          while ((line = input.readLine()) != null) {
              if (!line.trim().equals("")) {
                  // keep only the process name
                  line = line.substring(1); // remove the leading double quote
                  processes = processes + line.substring(0, line.indexOf("\"")) + ","; // remove the trailing double quote and add a comma
              }
          }
          input.close();
        }
        catch (Exception err) {
          err.printStackTrace();
        }
        return processes;
    }

    
    /**
     * Determines if the given IBAN is valid based on the check digit.
     * To validate the checksum:
     * 1. Check that the total IBAN length is correct as per the country. If not, the IBAN is invalid.
     * 2. Move the four initial characters to the end of the string.
     * 3. Replace the letters in the string with digits, expanding the string as necessary, such that A=10, B=11 and Z=35.
     * 4. Convert the string to an integer and mod-97 the entire number.
     * If the remainder is 1 you have a valid IBAN number.
     * @param iban
     * @return boolean indicating if specific IBAN has a valid check digit
     */
    public static Boolean isCheckDigitValid(String iban) {
        if (null == iban) return false;
        iban = iban.replaceAll(" ", "");
        if (iban.length() < 4) return false;
        int validIBANLength = getValidIBANLength(iban);
        if (iban.length() != validIBANLength) return false;
        BigInteger numericIBAN = getNumericIBAN(iban, false);
        int checkDigit = numericIBAN.mod(new BigInteger("97")).intValue();
        return checkDigit == 1;
    }

    
    /**
     * Get the IP address of the local machine
     * @return String with the IP address of the local machine
     */
    public static String getIP() {
        String ipAddr;
        try {
            InetAddress addr = InetAddress.getLocalHost();
            ipAddr = addr.getHostAddress();
        } catch (UnknownHostException e) {
            ipAddr = "IP Address could not be determined: " + e.getMessage();
        }        
        return ipAddr;
    }
    
    /**
     * Get the host name of the local machine
     * @return String with the host name of the local machine
     */
    public static String getHostName() {
        String hostName;
        try {
            InetAddress addr = InetAddress.getLocalHost();
            hostName = addr.getHostName();
        } catch (UnknownHostException e) {
            hostName = "Hostname could not be determined: " + e.getMessage();
        }        
        return hostName;
    }    
    
    /**
     * Using the IBAN.properties file gets the valid fixed length value for a country code.
     * Only uses the first 2 characters of the given string.
     * @param countryCode
     * @return int valid IBAN length for the specified country code
     */
    public static int getValidIBANLength(String countryCode) {
        String code = countryCode.substring(0,2).toUpperCase();
        String length = null;
        try {
            length = ResourceBundle.getBundle(UDF.class.getCanonicalName()).getString("length."+code);
        } catch(Exception e) {
            return -1; // the country code could not be found
        }
        if (length == null) return -1;
        return Integer.valueOf(length).intValue();
    }
    
    private static BigInteger getNumericIBAN(String iban, boolean isCheckDigitAtEnd) {
        String endCheckDigitIBAN = iban;
        if (!isCheckDigitAtEnd) {
            //Move first four characters to end of string to put check digit at end
            endCheckDigitIBAN = iban.substring(4) + iban.substring(0, 4);
        }
        StringBuffer numericIBAN = new StringBuffer();
        for (int i = 0; i < endCheckDigitIBAN.length(); i++) {
            if (Character.isDigit(endCheckDigitIBAN.charAt(i))) {
                numericIBAN.append(endCheckDigitIBAN.charAt(i));
            } else {
                numericIBAN.append(10 + getAlphabetPosition(endCheckDigitIBAN.charAt(i)));
            }
        }
        return new BigInteger(numericIBAN.toString());
    }

    private static int getAlphabetPosition(char letter) {
        return Character.valueOf(Character.toUpperCase(letter)).compareTo(Character.valueOf('A'));
    }


    /**
     * Use the SHA-256 algorithm and an input string to generate a hash code.
     * @param inputString
     * @return hash code
     */
    public static String getSHA256(String password) {
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            md.update(password.getBytes());
            byte[] byteData = md.digest();
            
            StringBuffer sb = new StringBuffer();
            for (int i = 0; i < byteData.length; i++) {
                sb.append(Integer.toString((byteData[i] & 0xff) + 0x100, 16).substring(1));
            }
            return sb.toString();
        } catch (NoSuchAlgorithmException ex) {
            return "Unexpected getSHA256 error: "+ex.getMessage();
        }
    }
    
    /**
     * @param url to be encoded
     * @return encoded url
     */
    public static String getEncodedUrl(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        } catch (Exception ex) {
            return "Unexpected URL encoding error: "+ex.getMessage();
        }
    }    
    
}
